package utils

import (
	"bytes"
	"encoding/json"
	"fmt"
	"jobarranger2/src/libs/golibs/common"
	"path/filepath"
	"strconv"
	"strings"
	"unicode"
)

func RawJsonToMapStringAny(from json.RawMessage) (map[string]any, error) {
	var err error
	var to map[string]any

	dec := json.NewDecoder(bytes.NewReader(from))
	dec.UseNumber()
	if err := dec.Decode(&to); err != nil {
		err = fmt.Errorf("Decode() failed, err: %w", err)
		return to, err
	}
	return to, err
}

// Decode Login Password
func DecodePassword(dPasswd string) string {
	dFlag := "1"
	key := "199907"
	var passwd string

	// check if dPasswd should be decoded
	if len(dPasswd) > 0 && dPasswd[0] == dFlag[0] {
		j, k := 0, 0
		dec := make([]byte, 0, len(dPasswd))

		// step 1: decode hex -> char and xor with key
		for kk := 1; kk < len(dPasswd); kk++ {
			if kk%2 == 0 { // even index
				hexStr := "0x" + string([]byte{dPasswd[kk-1], dPasswd[kk]})
				x16toX10, err := strconv.ParseInt(hexStr, 0, 64)
				if err != nil {
					return dPasswd // fallback
				}
				decodedChar := byte(x16toX10) ^ key[j]
				dec = append(dec, decodedChar)
				j++
				k++
				if j == len(key) {
					j = 0
				}
			}
		}

		// step 2: split at '|'
		decStr := string(dec)
		searchIndex := -1
		for i, c := range decStr {
			if c == '|' {
				searchIndex = i
				break
			}
		}

		if searchIndex == -1 {
			// no '|' found → fallback
			passwd = dPasswd
		} else {
			dNum := decStr[:searchIndex]

			// validate dNum is numeric
			isNumeric := true
			for _, c := range dNum {
				if !unicode.IsDigit(c) {
					isNumeric = false
					break
				}
			}

			if isNumeric {
				numVal, _ := strconv.Atoi(dNum)
				if numVal == len(decStr)-searchIndex-1 {
					// extract substring after '|'
					passwd = decStr[searchIndex+1:]
				} else {
					passwd = dPasswd
				}
			} else {
				passwd = dPasswd
			}
		}
	} else {
		passwd = dPasswd
	}

	return passwd
}

func GetFirstFileSource(files []common.FileTransfer) (string, error) {
	if len(files) == 0 {
		return "", fmt.Errorf("empty transfer files")
	}
	if files[0].Source == "" {
		return "", fmt.Errorf("source file is empty")
	}
	return files[0].Source, nil

}

// ParseTransactionFileName parses a transaction filename into its component parts.
// Expected format:
//
//	<InnerJobnetID>_<JobnetID>_<InnerJobID>_<Timestamp>.<Extension>
//
// Example:
//
//	1500000000000000275_MYJOBNET_223_20250912145641_612638856.json
func ParseTransactionFileName(filePath string) (*common.EventFileNameParts, error) {
	filename := filepath.Base(filePath)

	ext := filepath.Ext(filename)
	if ext == "" {
		return nil, fmt.Errorf("missing file extension in filename: %s", filename)
	}

	base := strings.TrimSuffix(filename, ext)
	parts := strings.Split(base, "_")
	if len(parts) < 5 {
		return nil, fmt.Errorf("invalid filename format: %s", filename)
	}

	// Parse InnerJobnetID
	innerJobnetID, err := strconv.ParseUint(parts[0], 10, 64)
	if err != nil {
		return nil, fmt.Errorf("invalid InnerJobnetID [%s]: %w", parts[0], err)
	}

	// Parse InnerJobID
	innerJobID, err := strconv.ParseUint(parts[len(parts)-3], 10, 64)
	if err != nil {
		return nil, fmt.Errorf("invalid InnerJobID [%s]: %w", parts[len(parts)-2], err)
	}

	// Parse JobnetID
	jobnetID := strings.Join(parts[1:len(parts)-3], "")

	// Parse Timestamp
	timestamp := strings.Join(parts[len(parts)-2:], "_")

	return &common.EventFileNameParts{
		InnerJobnetID: innerJobnetID,
		JobnetID:      jobnetID,
		InnerJobID:    innerJobID,
		Timestamp:     timestamp,
		Extension:     strings.TrimPrefix(ext, "."),
		FilePath:      filePath,
	}, nil
}
