/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/
package utils

import (
	"fmt"
	"jobarranger2/src/libs/golibs/logger/logger"
	"os"
	"time"

	"golang.org/x/sys/windows"
)

func getWindowsErrno(err error) (windows.Errno, bool) {
	switch e := err.(type) {
	case windows.Errno:
		return e, true

	case *os.PathError:
		if errno, ok := e.Err.(windows.Errno); ok {
			return errno, true
		}

	case *os.LinkError:
		if errno, ok := e.Err.(windows.Errno); ok {
			return errno, true
		}
	}

	return 0, false
}

func OpenFileWithLockRetry(path string, retryCount int) (*os.File, error) {
	funcName := "OpenFileWithLockRetry"

	var lastErr error
	var lastErrno windows.Errno

	for i := 1; i <= retryCount; i++ {
		f, err := os.OpenFile(path, os.O_RDWR, 0)
		if err == nil {
			return f, nil
		}

		lastErr = err
		if errno, ok := getWindowsErrno(err); ok {
			lastErrno = errno

			// ERROR_SHARING_VIOLATION
			if errno == windows.ERROR_SHARING_VIOLATION {
				logger.JaLog(
					"JAUTILS400007",
					logger.Logging{},
					funcName, path, i, retryCount,
				)

				time.Sleep(1 * time.Second)
				continue
			}

			// Other Windows errno → fail fast
			return nil, fmt.Errorf(
				"failed to open file. errno: %d (%s), path: %s, error: %w",
				int(errno), errno.Error(), path, err,
			)
		}
	}

	return nil, fmt.Errorf(
		"failed to open file after retries. path: %s, retryCount: %d, last errno: %d, last error: %w",
		path, retryCount, int(lastErrno), lastErr,
	)
}

func MoveFileWithLockRetry(src, dst string, retryCount int) error {
	funcName := "MoveFileWithLockRetry"

	var lastErr error
	var lastErrno windows.Errno

	for i := 1; i <= retryCount; i++ {
		err := os.Rename(src, dst)
		if err == nil {
			return nil
		}

		lastErr = err
		if errno, ok := getWindowsErrno(err); ok {
			lastErrno = errno

			// ERROR_SHARING_VIOLATION
			if errno == windows.ERROR_SHARING_VIOLATION {
				logger.JaLog(
					"JAUTILS400008",
					logger.Logging{},
					funcName, src, dst, i, retryCount,
				)

				time.Sleep(1 * time.Second)
				continue
			}

			// Other Windows error → fail fast
			return fmt.Errorf(
				"failed to rename file. errno: %d (%s), src: %s, dst: %s, error: %w",
				int(errno), errno.Error(), src, dst, err,
			)
		}
	}

	return fmt.Errorf(
		"failed to rename file after retries. src: %s, dst: %s, retryCount: %d, last errno: %d, last error: %w",
		src, dst, retryCount, int(lastErrno), lastErr,
	)
}
