/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/
package event

import (
	"jobarranger2/src/libs/golibs/common"
	"jobarranger2/src/libs/golibs/eventcore"
	"jobarranger2/src/libs/golibs/logger/logger"
)

// CreateNextEvent handles the creation of a JSON event file and sends it via a Unix domain socket.
//
// Parameters:
// - data:         EventData struct containing event details to be processed
// - innerJobnetId: uint64 representing the ID of the inner job network
// - jobnetId:      string representing the parent job network ID
// - innerJobId:    uint64 representing the ID of the job inside the inner jobnet
//
// Behavior:
// - It first generates a JSON file from the event data by calling CreateJsonFile().
// - Then, it sends the event data over a Unix domain socket to another process.
//
// Usage:
//
//	err := CreateNextEvent(eventData, 150**1, "jobnet_1", 1234)
//	if err != nil {
//	    log.Fatalf("failed to create json file: %v", err)
//	}
func CreateNextEvent(data common.EventData, innerJobnetId uint64, jobnetId string, innerJobId uint64) error {
	funcName := "CreateNextEvent"

	result := eventcore.CreateEventCore(&data, innerJobnetId, jobnetId, innerJobId)

	if result.Err != nil {
		return result.Err
	}
	logger.JaLog("JAUTILS000001", logger.Logging{}, funcName, data.Transfer.Files[0].Source, data.Event.Name)

	if result.UDSSuccess {
		logger.JaLog("JAUTILS400001", logger.Logging{}, funcName, data.Event.Name, innerJobnetId, innerJobId)
	} else {
		logger.JaLog("JAUTILS300001", logger.Logging{}, funcName, data.Event.Name, innerJobnetId, innerJobId, result.UDSErrMsg)
	}

	return nil
}
